<!-- =========================================================================
 Copyright (c) 2020-2024 XMLmind Software. All rights reserved.
 
 Author: Hussein Shafie
 
 This file is part of the XMLmind W2X project.
 For conditions of distribution and use, see the accompanying legal.txt file.
========================================================================== -->

<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:saxon="http://icl.com/saxon"
  xmlns:exsl="http://exslt.org/common"
  xmlns:p="java:com.xmlmind.w2x.processor.Processor"
  xmlns="http://docbook.org/ns/docbook"
  xmlns:d="http://docbook.org/ns/docbook"
  extension-element-prefixes="saxon exsl"
  exclude-result-prefixes="saxon exsl p">

<xsl:param name="section-depth" select="1" />

<xsl:param name="output-path" select="''" />

<!-- Assembly and topics are generally both given a '.xml' file extension so
     there may be an output file name conflict if the name of the assembly
     output file happens to be equal to the ID of a topic. -->
<xsl:param name="output-name" select="''" />

<xsl:param name="topic-path" select="''" />

<xsl:param name="add-index" select="'yes'" />

<xsl:output method="xml" encoding="UTF-8" indent="no"/>

<!-- Generate an assembly ================================================ -->

<xsl:template match="/*"> <!-- Always a book because of
                               "-p transform.hierarchyName book"
                               in Processor.java. -->
  <assembly>
    <xsl:call-template name="addDocBookVersion" />
    
    <resources>
      <xsl:apply-templates select="./d:chapter|./d:section" mode="resource" />
    </resources>
    
    <structure renderas="book">
      <xsl:copy-of select="./@xml:lang" />
      
      <xsl:variable name="infos" select="./d:info/*"/>
      <xsl:if test="count($infos) &gt; 0">
        <xsl:element name="d:merge">
          <xsl:for-each select="$infos">
            <xsl:copy-of select="." />
          </xsl:for-each>
        </xsl:element>
      </xsl:if>
      
      <xsl:apply-templates select="./d:chapter|./d:section" />
      
      <xsl:if test="$add-index = 'yes' and //d:indexterm">
        <module renderas="index" />
      </xsl:if>
    </structure>
  </assembly>
</xsl:template>

<xsl:template name="addDocBookVersion">
  <xsl:variable name="version">
    <xsl:choose>
      <!-- Take it from the DocBook 5.1+ book being transformed. -->
      <xsl:when test="/*/@version">
        <xsl:value-of select="/*/@version" />
      </xsl:when>
      <xsl:otherwise>5.1</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:attribute name="version"><xsl:value-of
    select="$version"/></xsl:attribute>
</xsl:template>

<!-- resource ========================= -->

<xsl:template match="d:chapter|d:section" mode="resource">
  <xsl:variable name="depth"
                select="count(ancestor::*[self::d:book or self::d:chapter or
                                          self::d:section])" />
  
  <xsl:if test="$depth &lt;= $section-depth">
    <xsl:variable name="topicId">
      <xsl:call-template name="topicId" />
    </xsl:variable>
    
    <xsl:variable name="topicFile">
      <xsl:call-template name="topicFilename">
        <xsl:with-param name="id" select="$topicId" />
      </xsl:call-template>
    </xsl:variable>

    <resource xml:id="{$topicId}" href="{$topicFile}" />
    
    <xsl:apply-templates select="./d:section" mode="resource" />
  </xsl:if>
</xsl:template>

<xsl:template name="topicId">
  <xsl:choose>
    <xsl:when test="@xml:id">
      <xsl:value-of select="@xml:id"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="generate-id()"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="topicFilename">
  <xsl:param name="id" />

  <xsl:variable name="id2">
    <xsl:choose>
      <xsl:when test="$id = $output-name">
        <xsl:value-of select="concat($id, '-', generate-id())" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$id" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  
  <xsl:variable name="topicDir">
    <xsl:call-template name="appendSlash">
      <xsl:with-param name="path" select="$topic-path" />
    </xsl:call-template>
  </xsl:variable>

  <xsl:value-of select="concat($topicDir, $id2, '.xml')" />
</xsl:template>

<xsl:template name="appendSlash">
  <xsl:param name="path" />

  <xsl:if test="$path != ''">
    <xsl:choose>
      <xsl:when test="substring($path, string-length($path)) = '/'">
        <xsl:value-of select="$path" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat($path, '/')" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>
</xsl:template>

<!-- module ========================= -->

<xsl:template match="d:chapter|d:section">
  <xsl:variable name="depth"
                select="count(ancestor::*[self::d:book or self::d:chapter or
                                          self::d:section])" />
  
  <xsl:if test="$depth &lt;= $section-depth">
    <!-- .xml file -->

    <xsl:variable name="topic">
      <topic>
        <xsl:copy-of select="/*/@xml:lang" /> <!-- Topmost xml:lang -->
        <xsl:copy-of select="@*" /> <!-- May override topmost xml:lang -->
        <xsl:call-template name="addDocBookVersion" />
        
        <xsl:choose>
          <xsl:when test="$depth = $section-depth">
            <!-- Copy all child elements, including child sections. -->
            <xsl:apply-templates select="./*" mode="copy"/>
          </xsl:when>
          <xsl:otherwise>
            <!-- Copy all child elements, except child sections. -->
            <xsl:apply-templates select="./*[not(self::d:section)]"
                                 mode="copy"/>
          </xsl:otherwise>
        </xsl:choose>
      </topic>
    </xsl:variable>

    <xsl:variable name="topicId">
      <xsl:call-template name="topicId" />
    </xsl:variable>
    
    <xsl:variable name="topicFile">
      <xsl:call-template name="topicFilename">
        <xsl:with-param name="id" select="$topicId" />
      </xsl:call-template>
    </xsl:variable>

    <xsl:call-template name="writeTopic">
      <xsl:with-param name="file" select="$topicFile" />
      <xsl:with-param name="topic" select="$topic" />
    </xsl:call-template>

    <!-- module -->
    
    <xsl:variable name="topicName">
      <xsl:choose>
        <xsl:when test="$depth = 1">chapter</xsl:when>
        <xsl:otherwise>section</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <module renderas="{$topicName}" resourceref="{$topicId}">
      <xsl:apply-templates select="./d:section" />
    </module>
  </xsl:if>
</xsl:template>

<!-- Generate topics ===================================================== -->

<!-- imagedata ========================= -->

<xsl:template match="d:imagedata[@fileref and 
                                 not(starts-with(@fileref, 'file:/') or 
                                     contains(@fileref, '://'))]"
              mode="copy">
  <xsl:choose>
    <xsl:when test="$topic-path != ''">
      <xsl:variable name="topicDir">
        <xsl:call-template name="appendSlash">
          <xsl:with-param name="path" select="$topic-path" />
        </xsl:call-template>
      </xsl:variable>

      <xsl:variable name="assemblyDir">
        <xsl:call-template name="topicToAssemblyPath">
          <xsl:with-param name="path" select="$topicDir" />
        </xsl:call-template>
      </xsl:variable>

      <imagedata fileref="{concat($assemblyDir, @fileref)}">
        <xsl:apply-templates select="@*[local-name() != 'fileref']|node()" 
                             mode="copy" />
      </imagedata>
    </xsl:when>

    <xsl:otherwise>
      <xsl:copy>
        <xsl:apply-templates select="@*|node()" mode="copy" />
      </xsl:copy>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="topicToAssemblyPath">
  <xsl:param name="path" />

  <xsl:if test="contains($path, '/')">
    <xsl:variable name="tail">
      <xsl:call-template name="topicToAssemblyPath">
        <xsl:with-param name="path" select="substring-after($path, '/')" />
      </xsl:call-template>
    </xsl:variable>

    <xsl:value-of select="concat('../', $tail)" />
  </xsl:if>
</xsl:template>

<!-- Topic elements other than imagedata ========================= -->

<xsl:template match="@*|node()" mode="copy">
  <xsl:copy>
    <xsl:apply-templates select="@*|node()" mode="copy" />
  </xsl:copy>
</xsl:template>

<!-- writeTopic ========================= -->

<xsl:template name="writeTopic">
  <xsl:param name="file" />
  <xsl:param name="topic" />

  <xsl:if test="not(function-available('exsl:node-set'))">
    <xsl:message terminate="yes">
      <xsl:text>Function 'exsl:node-set' not implemented by </xsl:text>
      <xsl:value-of select="system-property('xsl:vendor')"/>
      <xsl:text>.</xsl:text>
    </xsl:message>
  </xsl:if>

  <xsl:variable name="savedTopic" select="exsl:node-set($topic)/*" />

  <xsl:variable name="outputDir">
    <xsl:call-template name="appendSlash">
      <xsl:with-param name="path" select="$output-path" />
    </xsl:call-template>
  </xsl:variable>

  <xsl:variable name="href">
    <xsl:value-of select="concat($outputDir, $file)" />
  </xsl:variable>

  <xsl:choose>
    <xsl:when test="element-available('exsl:document')">
      <exsl:document href="{$href}" method="xml" encoding="UTF-8" indent="no">
        <xsl:copy-of select="$savedTopic"/>
      </exsl:document>
    </xsl:when>

    <xsl:when test="element-available('saxon:output')">
      <!-- Saxon 6.5 interprets the href attribute of <xsl:document> as a
           filename relative to the current working directory. You can use an
           absolute filename if you want, but not an absolute URI. This is a
           documented restriction, not a bug. -->

      <xsl:variable name="href2">
        <xsl:choose>
          <xsl:when test="starts-with($href, 'file:/')">
            <xsl:value-of select="p:fileURLToPath($href)" />
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="$href" />
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>

      <saxon:output href="{$href2}" method="xml" encoding="UTF-8" indent="no">
        <xsl:copy-of select="$savedTopic"/>
      </saxon:output>
    </xsl:when>

    <xsl:otherwise>
      <xsl:message terminate="yes">
        <xsl:text>Multiple result documents not implemented by </xsl:text>
        <xsl:value-of select="system-property('xsl:vendor')"/>
        <xsl:text>.</xsl:text>
      </xsl:message>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

</xsl:stylesheet>
