<!-- =========================================================================
 Copyright (c) 2017-2024 XMLmind Software. All rights reserved.
 
 Author: Hussein Shafie
 
 This file is part of the XMLmind W2X project.
 For conditions of distribution and use, see the accompanying legal.txt file.
========================================================================== -->

<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns="http://www.w3.org/1999/xhtml"
  xmlns:h="http://www.w3.org/1999/xhtml"
  xmlns:mml="http://www.w3.org/1998/Math/MathML"
  exclude-result-prefixes="h mml">

<!-- a @class=role-xref ================================================== -->

<xsl:param name="generate-xref-text" select="'yes'"/>

<xsl:key name="captions" 
  match="h:div[starts-with(@class, 'role-section')]
              [./*[1][self::h:h1 or self::h:h2 or self::h:h3 or 
                      self::h:h4 or self::h:h5 or self::h:h6]] |
         h:h1 | h:h2 | h:h3 | h:h4 | h:h5 | h:h6 |
         h:p[starts-with(@class, 'role-bridgehead')] |
         h:table[./h:caption] |
         h:caption |
         h:div[starts-with(@class, 'role-figure')]
              [./h:p[@class='role-figcaption']] |
         h:p[@class='role-figcaption']" 
  use="@id"/>

<xsl:template match="h:a[starts-with(@href, '#')][@class='role-xref']">
  <xsl:element name="a">
    <xsl:choose>
      <xsl:when test="$generate-xref-text = 'yes'">
        <xsl:apply-templates select="@*[local-name() != 'title']"/>

        <xsl:variable name="caption" 
                      select="key('captions', substring-after(@href, '#'))"/>
        <xsl:choose>
          <xsl:when test="$caption and 
                          ($caption/self::h:div or $caption/self::h:table)">
            <xsl:value-of select="normalize-space($caption/*[1])"/>
          </xsl:when>

          <xsl:when test="$caption">
            <xsl:value-of select="normalize-space($caption)"/>
          </xsl:when>

          <xsl:otherwise>
            <xsl:variable name="title" select="normalize-space(@title)"/>
            <xsl:choose>
              <xsl:when test="$title != ''">
                <xsl:value-of select="$title"/>
              </xsl:when>
              <xsl:otherwise>???</xsl:otherwise>
            </xsl:choose>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>

      <xsl:otherwise>
        <xsl:apply-templates select="@*|node()"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:element>
</xsl:template>

<!-- a @class=role-footnote-ref,role-endnote-ref ========================= -->

<xsl:param name="number-footnotes" select="'yes'"/>

<xsl:param name="footnote-number-format" select="'[1]'"/>

<xsl:template match="h:div[@class='role-footnotes' or @class='role-endnotes']">
  <xsl:apply-templates/>
</xsl:template>

<xsl:key name="footnotes" 
         match="h:div[@class='role-footnote' or @class='role-endnote']" 
         use="@id"/>

<xsl:template match="h:div[@class='role-footnote' or @class='role-endnote']">
  <xsl:if test="$number-footnotes = 'yes' and 
                not(preceding::h:div[@class='role-footnote' or 
                                     @class='role-endnote'])">
    <xsl:element name="hr">
      <xsl:attribute name="class">role-footnote-separator</xsl:attribute>
    </xsl:element>
  </xsl:if>

  <xsl:choose>
    <xsl:when test="$number-footnotes = 'yes'">
      <xsl:element name="div">
        <xsl:apply-templates select="@*"/>

        <xsl:variable name="para" select="./*[position() = 1 and self::h:p]"/>
        <xsl:choose>
          <xsl:when test="$para">
            <xsl:for-each select="$para">
              <xsl:element name="p">
                <xsl:apply-templates select="@*"/>
                <xsl:call-template name="footnoteNumber"/>
                <xsl:apply-templates/>
              </xsl:element>
            </xsl:for-each>

            <xsl:apply-templates select="./*[position() > 1]"/>
          </xsl:when>

          <xsl:otherwise>
            <xsl:call-template name="footnoteNumber"/>
            <xsl:apply-templates/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:element>
    </xsl:when>

    <xsl:otherwise>
      <xsl:element name="div">
        <xsl:apply-templates select="@*|node()"/>
      </xsl:element>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="footnoteNumber">
  <xsl:element name="sup">
    <xsl:attribute name="class">role-footnote-number</xsl:attribute>
    <xsl:number format="{$footnote-number-format}" level="any"
                count="h:div[@class='role-footnote' or @class='role-endnote']"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:a[starts-with(@href, '#')][@class='role-footnote-ref' or 
                                                  @class='role-endnote-ref']">
  <xsl:element name="a">
    <xsl:choose>
      <xsl:when test="$number-footnotes = 'yes'">
        <xsl:apply-templates select="@*[local-name() != 'title']"/>

        <xsl:variable name="footnote"
                      select="key('footnotes', substring-after(@href, '#'))"/>
        <xsl:choose>
          <xsl:when test="$footnote">
            <xsl:for-each select="$footnote">
              <xsl:call-template name="footnoteNumber"/>
            </xsl:for-each>
          </xsl:when>
          <xsl:otherwise>???</xsl:otherwise>
        </xsl:choose>
      </xsl:when>

      <xsl:otherwise>
        <xsl:apply-templates select="@*|node()"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:element>
</xsl:template>

<!-- p @class=role-bridgehead[123456] ==================================== -->

<xsl:template match="h:p[@class='role-bridgehead1']">
  <xsl:element name="h1">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:p[@class='role-bridgehead2']">
  <xsl:element name="h2">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:p[@class='role-bridgehead3']">
  <xsl:element name="h3">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:p[@class='role-bridgehead4']">
  <xsl:element name="h4">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:p[@class='role-bridgehead5']">
  <xsl:element name="h5">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:p[@class='role-bridgehead6']">
  <xsl:element name="h6">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<!-- span @class=role-index-term,-start,-end ============================= -->

<xsl:param name="discard-index-terms" select="'yes'"/>

<xsl:template match="h:span[@class='role-index-term' or 
                            @class='role-index-term-start' or 
                            @class='role-index-term-end']">
  <xsl:if test="$discard-index-terms != 'yes'">
    <xsl:element name="span">
      <xsl:apply-templates select="@*|node()"/>
    </xsl:element>
  </xsl:if>
</xsl:template>

<!-- ruby ================================================================ -->

<!-- ruby is allowed in XHTML 1.1 and XHTML 5 only.
     However the ruby generated by MS-Word "Save as HTML" is not valid. -->

<xsl:template match="h:ruby">
  <xsl:element name="span">
    <xsl:apply-templates select="@*"/>
    <xsl:value-of select="."/>
  </xsl:element>
</xsl:template>

<!-- tr ================================================================== -->

<xsl:template match="h:tr">
  <xsl:element name="tr">
    <xsl:apply-templates select="@*|node()"/>

    <xsl:if test="count(./h:th | ./h:td) = 0">
      <!-- In some rare cases, a row may be empty. -->
      <xsl:element name="td"/>
    </xsl:if>
  </xsl:element>
</xsl:template>

<!-- MathML ============================================================== -->
<!-- In HTML mode, we need mml:math to use the default namespace. -->

<xsl:template match="mml:math">
  <xsl:apply-templates select="." mode="math"/>
</xsl:template>

<xsl:template match="mml:*" mode="math">
  <xsl:element name="{local-name()}" xmlns="http://www.w3.org/1998/Math/MathML">
    <xsl:apply-templates select="@*|node()" mode="math"/>
  </xsl:element>
</xsl:template>

<xsl:template match="@*|node()" mode="math">
  <xsl:copy><xsl:apply-templates select="@*|node()" mode="math"/></xsl:copy>
</xsl:template>

<!-- @lang and @xml:lang ================================================= -->

<xsl:param name="add-xml-lang" select="'no'"/>

<xsl:template match="@lang">
  <xsl:copy/>

  <xsl:if test="$add-xml-lang = 'yes'">
    <xsl:variable name="lang" select="."/>
    <xsl:for-each select="parent::h:*[not(@xml:lang)]">
      <xsl:attribute name="xml:lang"><xsl:value-of
        select="$lang"/></xsl:attribute>
    </xsl:for-each>
  </xsl:if>
</xsl:template>

<!-- Copy as is everything else ========================================== -->

<xsl:template match="@*|node()">
  <xsl:copy><xsl:apply-templates select="@*|node()"/></xsl:copy>
</xsl:template>

<!-- Utilities =========================================================== -->

<xsl:param name="style-with-class" select="'no'"/>

<xsl:template name="addClassOrStyle">
  <xsl:param name="class" select="''"/>
  <xsl:param name="style" select="''"/>
  
  <xsl:choose>
    <xsl:when test="$style-with-class = 'yes'">
      <xsl:call-template name="addClass">
        <xsl:with-param name="class" select="$class"/>
      </xsl:call-template>

      <xsl:for-each select="@style">
        <xsl:copy/>
      </xsl:for-each>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="addStyle">
        <xsl:with-param name="style" select="$style"/>
      </xsl:call-template>

      <xsl:for-each select="@class">
        <xsl:copy/>
      </xsl:for-each>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="addStyle">
  <xsl:param name="style" select="''"/>

  <xsl:variable name="style1" select="normalize-space(@style)"/>
  <xsl:variable name="style2" select="normalize-space($style)"/>

  <xsl:if test="$style1 != '' or $style2 != ''">
    <xsl:attribute name="style">
      <xsl:choose>
        <xsl:when test="$style1 != '' and 
                        (not(contains($style1, ';')) or
                         substring-after($style1, ';') != '')">
          <xsl:value-of
            select="normalize-space(concat($style1, ';', $style2))"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="normalize-space(concat($style1, $style2))"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:attribute>
  </xsl:if>
</xsl:template>

<xsl:template name="addClass">
  <xsl:param name="class" select="''"/>

  <xsl:variable name="class1" select="normalize-space(@class)"/>
  <xsl:variable name="class2" select="normalize-space($class)"/>

  <xsl:if test="$class1 != '' or $class2 != ''">
    <xsl:attribute name="class">
      <xsl:value-of select="normalize-space(concat($class1, ' ', $class2))"/>
    </xsl:attribute>
  </xsl:if>
</xsl:template>

</xsl:stylesheet>
