<!-- =========================================================================
 Copyright (c) 2017-2019 XMLmind Software. All rights reserved.
 
 Author: Hussein Shafie
 
 This file is part of the XMLmind W2X project.
 For conditions of distribution and use, see the accompanying legal.txt file.
========================================================================== -->

<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns="http://www.w3.org/1999/xhtml"
                xmlns:h="http://www.w3.org/1999/xhtml"
                exclude-result-prefixes="h">

<!-- =========================================================================
     Cannot be translated:
     * object (very different attributes).

     Differences with the Strict DTD (other than new elements and 
     new attributes):
     * Removed: acronym, big, tt.
     * Restored (like in the Transitional DTD): s, u, iframe.
     * a has no @name, @charset, @coords, @rev, @shape 
       but @target is back everywhere.
       Note that some other elements like form, fieldset keep their @name.
     * Restored (like the Transitional DTD): ol/@start, @type; ol/li/@value.
     * head cannot contain: object.
     * Has flow content (like the Transitional DTD): blockquote, body.
     * Other different content models: address, button, caption, form, map, 
       object, table (cannot directly contain col).

     Tables: all presentation attributes (except table/@border which
     has been simplified: 1, "" or absent) have been removed:
     * table: @cellpadding, @cellspacing, @frame, @rules, @width.
     * colgroup, col: @width.
     * colgroup, col, tbody, thead, tfoot, tr, td, th: 
       @align, @char, @charoff, @valign. 
     Moreover table has no @summary and th, td have no: @abbr, @axis, @scope.
========================================================================== -->

<xsl:import href="xhtml_common.xslt"/>
<xsl:import href="xhtml_strict_common.xslt"/>
<xsl:import href="xhtml_body_flow.xslt"/>

<xsl:output method="xml" encoding="UTF-8" indent="no"/>

<xsl:template match="/">
  <xsl:text disable-output-escaping="yes">&lt;!DOCTYPE html&gt;</xsl:text>

  <xsl:apply-templates/>
</xsl:template>

<!-- span @class=role-mark =============================================== -->

<xsl:template match="h:span[@class='role-mark']">
  <xsl:element name="mark">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<!-- div @class=role-figure ============================================== -->

<xsl:template match="h:div[@class='role-figure']">
  <xsl:element name="figure">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:p[@class='role-figcaption']">
  <xsl:element name="figcaption">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<!-- div @class=role-equation ============================================ -->

<xsl:template match="h:div[@class='role-equation']">
  <xsl:element name="figure">
    <xsl:apply-templates select="@*|node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:p[@class='role-equation-caption']">
  <xsl:element name="figcaption">
    <xsl:apply-templates select="@*|node()"/>
  </xsl:element>
</xsl:template>

<!-- div @class=role-section[123456] ===================================== -->

<xsl:template match="h:div[starts-with(@class, 'role-section')]">
  <xsl:element name="section">
    <xsl:apply-templates select="@*[local-name() != 'class'] | node()"/>
  </xsl:element>
</xsl:template>

<!-- ===================================================================== -->

<xsl:template match="h:acronym">
  <xsl:element name="abbr">
    <xsl:apply-templates select="@*|node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:a/@name">
  <xsl:call-template name="nameToId"/>
</xsl:template>

<xsl:template match="@target">
  <xsl:copy/>
</xsl:template>

<xsl:template match="h:form/@name | h:fieldset/@name">
  <xsl:copy/>
</xsl:template>

<xsl:template match="h:big">
  <xsl:element name="span">
    <xsl:call-template name="addClassOrStyle">
      <xsl:with-param name="class" select="'big'"/>
      <xsl:with-param name="style" select="'font-size:larger;'"/>
    </xsl:call-template>

    <xsl:apply-templates select="@*[local-name() != 'class' and
                                    local-name() != 'style'] | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:ol">
  <xsl:element name="ol">
    <!-- Note that @compact is anyway discarded by xhtml_strict_common. -->
    <xsl:apply-templates select="@* | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:ol/h:li/@value">
  <xsl:copy/>
</xsl:template>

<xsl:template match="h:s">
  <xsl:element name="s">
    <xsl:apply-templates select="@* | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:tt">
  <xsl:element name="span">
    <xsl:call-template name="addClassOrStyle">
      <xsl:with-param name="class" select="'tt'"/>
      <xsl:with-param name="style" select="'font-family:monospace;'"/>
    </xsl:call-template>

    <xsl:apply-templates select="@*[local-name() != 'class' and
                                    local-name() != 'style'] | node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:u">
  <xsl:element name="u">
    <xsl:apply-templates select="@* | node()"/>
  </xsl:element>
</xsl:template>

<!-- Fix @datetime ======================================================= -->

<xsl:template match="h:ins/@datetime | h:del/@datetime">
  <xsl:choose>
    <!-- Example: 2001-04-05T11:37 -->
    <xsl:when test="string-length(.) = 16">
      <xsl:attribute name="datetime">
        <xsl:value-of select="concat(., ':00')"/>
      </xsl:attribute>
    </xsl:when>
    <xsl:otherwise><xsl:copy/></xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Simply discard these relatively unimportant attributes. -->

<xsl:template match="h:a/@charset | h:a/@coords | h:a/@rev |  h:a/@shape |
                     h:area/@nohref |
                     h:form/@accept |
                     h:head/@profile |
                     h:img/@longdesc |
                     h:link/@charset | h:link/@rev |
                     h:meta/@scheme |
                     h:param/@type | h:param/@valuetype |
                     h:table/@cellpadding | h:table/@cellspacing |
                     h:table/@frame | h:table/@rules |
                     h:table/@summary |
                     h:td/@abbr | h:td/@axis | h:td/@scope |
                     h:th/@abbr | h:th/@axis | h:th/@scope"/>

<!-- Presentation attributes in tables =================================== -->

<xsl:template match="h:table | 
                     h:colgroup | h:col | 
                     h:tbody | h:thead | h:tfoot | 
                     h:tr | 
                     h:th | h:td">
  <xsl:element name="{name()}">
    <xsl:if test="self::h:table and @border != '0'">
      <xsl:attribute name="border">1</xsl:attribute>
    </xsl:if>

    <xsl:variable name="cellspacing">
      <xsl:if test="self::h:table and number(@cellspacing) &gt;= 0">
        <xsl:value-of select="@cellspacing"/>
      </xsl:if>
    </xsl:variable>

    <!-- table/@align specifies the align of the table itself. -->
    <xsl:variable name="align">
      <xsl:if test="(@align = 'center' or @align = 'right' or 
                     @align = 'left') and not(self::h:table)">
        <xsl:value-of select="@align"/>
      </xsl:if>
    </xsl:variable>

    <xsl:variable name="valign">
      <xsl:if test="@valign = 'top' or @valign = 'bottom' or 
                    @valign = 'middle' or @valign = 'baseline'">
        <xsl:value-of select="@valign"/>
      </xsl:if>
    </xsl:variable>

    <xsl:variable name="width">
      <xsl:choose>
        <xsl:when test="substring-before(@width, '%') != ''">
          <xsl:value-of select="@width"/>
        </xsl:when>
        <xsl:when test="@width != ''">
          <xsl:value-of select="concat(@width, 'px')"/>
        </xsl:when>
      </xsl:choose>
    </xsl:variable>

    <xsl:variable name="style">
      <xsl:if test="$cellspacing != ''">
        <xsl:value-of select="concat('border-spacing:', $cellspacing, 'px;')"/>
      </xsl:if>
      <xsl:if test="$align != ''">
        <xsl:value-of select="concat('text-align:', $align, ';')"/>
      </xsl:if>
      <xsl:if test="$valign != ''">
        <xsl:value-of select="concat('vertical-align:', $valign, ';')"/>
      </xsl:if>
      <xsl:if test="$width != ''">
        <xsl:value-of select="concat('width:', $width, ';')"/>
      </xsl:if>
    </xsl:variable>

    <xsl:call-template name="addStyle">
      <xsl:with-param name="style" select="$style"/>
    </xsl:call-template>

    <xsl:apply-templates select="@*[local-name() != 'border' and
                                    local-name() != 'cellspacing' and
                                    local-name() != 'style' and
                                    local-name() != 'align' and
                                    local-name() != 'valign' and
                                    local-name() != 'width']"/>

    <xsl:choose>
      <xsl:when test="self::h:table">
        <xsl:apply-templates select="./h:caption"/>
        <xsl:apply-templates select="./h:colgroup"/>
        <xsl:apply-templates select="./h:thead"/>
        <xsl:apply-templates select="./h:tbody"/>
        <!-- In HTML5, the tfoot, if any, is *after* tbody. -->
        <xsl:apply-templates select="./h:tfoot "/>
      </xsl:when>

      <xsl:when test="self::h:tr">
        <xsl:variable name="cells" select="./h:th | ./h:td"/>
        <xsl:choose>
          <xsl:when test="count($cells) = 0">
            <!-- In some rare cases, a row may be empty. -->
            <xsl:element name="td"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:apply-templates select="$cells"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>

      <xsl:otherwise>
        <xsl:apply-templates select="./node()"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:element>
</xsl:template>

</xsl:stylesheet>
