<!-- =========================================================================
 Copyright (c) 2017-2019 XMLmind Software. All rights reserved.
 
 Author: Hussein Shafie
 
 This file is part of the XMLmind W2X project.
 For conditions of distribution and use, see the accompanying legal.txt file.
========================================================================== -->

<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:exsl="http://exslt.org/common"
  xmlns:sets="http://exslt.org/sets"
  xmlns:saxon="http://icl.com/saxon"
  xmlns:h="http://www.w3.org/1999/xhtml"
  xmlns:mml="http://www.w3.org/1998/Math/MathML"
  exclude-result-prefixes="exsl sets saxon h mml">

<xsl:include href="common_params.xslt"/>
<xsl:include href="other_common.xslt"/>

<!-- As of DITA 1.3, supported by DITA CALS tables. -->
<xsl:param name="processCellRotate" select="'yes'"/>
<xsl:include href="tgroup.xslt"/>

<xsl:output method="xml" encoding="UTF-8" indent="no"/>

<xsl:param name="topic-type" select="'topic'"/>
<xsl:param name="single-topic" select="'no'"/>
<xsl:param name="root-topic-id" select="''"/>
<xsl:param name="pre-element-name" select="'pre'"/>

<!-- Examples: "p-Shortdesc", "p-Abstract". -->
<xsl:param name="shortdesc-class-name" select="''"/> 
<xsl:param name="shortdescClassName"> 
  <xsl:if test="$shortdesc-class-name != '' and 
                function-available('sets:difference')">
    <xsl:value-of select="$shortdesc-class-name"/>
  </xsl:if>
</xsl:param>

<xsl:template match="/">
  <xsl:choose>
    <xsl:when test="$topic-type = 'concept'">
      <xsl:text disable-output-escaping="yes">&lt;!DOCTYPE concept PUBLIC "-//OASIS//DTD DITA Concept//EN" "concept.dtd"&gt;</xsl:text>
    </xsl:when>
    <xsl:when test="$topic-type = 'generalTask'">
      <xsl:text disable-output-escaping="yes">&lt;!DOCTYPE task PUBLIC "-//OASIS//DTD DITA General Task//EN" "generalTask.dtd"&gt;</xsl:text>
    </xsl:when>
    <xsl:when test="$topic-type = 'task' or $topic-type = 'strictTask'">
      <xsl:text disable-output-escaping="yes">&lt;!DOCTYPE task PUBLIC "-//OASIS//DTD DITA Task//EN" "task.dtd"&gt;</xsl:text>
    </xsl:when>
    <xsl:when test="$topic-type = 'reference'">
      <xsl:text disable-output-escaping="yes">&lt;!DOCTYPE reference PUBLIC "-//OASIS//DTD DITA Reference//EN" "reference.dtd"&gt;</xsl:text>
    </xsl:when>
    <xsl:otherwise>
      <xsl:text disable-output-escaping="yes">&lt;!DOCTYPE topic PUBLIC "-//OASIS//DTD DITA Topic//EN" "topic.dtd"&gt;</xsl:text>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:apply-templates/>
</xsl:template>

<!-- html ================================================================ -->

<xsl:template match="h:html">
  <xsl:variable name="topicName">
    <xsl:call-template name="topicName"/>
  </xsl:variable>
  <xsl:element name="{$topicName}">
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:if test="not(@id)">
      <xsl:attribute name="id">
        <xsl:choose>
          <xsl:when test="$root-topic-id != '' and $root-topic-id != '???'">
            <xsl:value-of select="$root-topic-id"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="generateTopicId"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
    </xsl:if>

    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<xsl:template name="topicName">
  <xsl:choose>
    <xsl:when test="$topic-type = 'concept'">concept</xsl:when>
    <xsl:when test="$topic-type = 'task' or
                    $topic-type = 'strictTask' or
                    $topic-type = 'generalTask'">task</xsl:when>
    <xsl:when test="$topic-type = 'reference'">reference</xsl:when>
    <xsl:otherwise>topic</xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="generateTopicId">
  <xsl:value-of select="generate-id()"/>
</xsl:template>

<!-- head, title, meta =================================================== -->

<xsl:template match="h:head">
  <xsl:apply-templates select="./h:title"/>

  <xsl:variable name="subject" select="(./h:meta[@name = 'description'])[1]"/>
  <xsl:choose>
    <xsl:when test="$subject">
      <shortdesc>
        <xsl:value-of select="$subject/@content"/>
      </shortdesc>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="topicShortdesc"/>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:variable name="prologContent">
    <xsl:apply-templates select="./h:meta[@name = 'author']"/>

    <xsl:apply-templates select="(./h:meta[@name = 'dcterms.publisher'])[1]"/>

    <xsl:if test="./h:meta[@name = 'dcterms.created' or
                           @name = 'dcterms.modified']">
      <critdates>
        <xsl:apply-templates select="(./h:meta[@name = 'dcterms.created'])[1]"/>

        <xsl:apply-templates select="./h:meta[@name = 'dcterms.modified']"/>
      </critdates>
    </xsl:if>

    <xsl:if test="./h:meta[@name = 'keywords'] and 
                  function-available('saxon:tokenize')">
      <metadata>
        <xsl:apply-templates select="./h:meta[@name = 'keywords']"/>
      </metadata>
    </xsl:if>

    <xsl:apply-templates select="./h:meta[@name != 'description' and 
                                          @name != 'author' and 
                                          @name != 'dcterms.publisher' and 
                                          @name != 'dcterms.created' and 
                                          @name != 'dcterms.modified' and 
                                          @name != 'keywords']"/>
  </xsl:variable>

  <xsl:if test="not(function-available('exsl:node-set'))">
    <xsl:message terminate="yes">
      <xsl:text>Function 'exsl:node-set' not implemented by </xsl:text>
      <xsl:value-of select="system-property('xsl:vendor')"/>
      <xsl:text>.</xsl:text>
    </xsl:message>
  </xsl:if>

  <xsl:if test="exsl:node-set($prologContent)/*">
    <prolog>
      <xsl:copy-of select="$prologContent"/>
    </prolog>
  </xsl:if>
</xsl:template>

<xsl:template match="h:title |
                     h:div[starts-with(@class, 'role-section')]/h:h1 |
                     h:div[starts-with(@class, 'role-section')]/h:h2 |
                     h:div[starts-with(@class, 'role-section')]/h:h3 |
                     h:div[starts-with(@class, 'role-section')]/h:h4 |
                     h:div[starts-with(@class, 'role-section')]/h:h5 |
                     h:div[starts-with(@class, 'role-section')]/h:h6 |
                     h:p[@class = 'role-figcaption'] |
                     h:p[@class = 'role-equation-caption'] |
                     h:caption">
  <title>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </title>
</xsl:template>

<xsl:template match="h:meta[@name = 'author']">
  <author>
    <xsl:value-of select="@content"/>
  </author>
</xsl:template>

<xsl:template match="h:meta[@name = 'dcterms.publisher']">
  <publisher>
    <xsl:value-of select="@content"/>
  </publisher>
</xsl:template>

<xsl:template match="h:meta[@name = 'dcterms.created']">
  <created date="{@content}"/>
</xsl:template>

<xsl:template match="h:meta[@name = 'dcterms.modified']">
  <revised modified="{@content}"/>
</xsl:template>

<xsl:template match="h:meta[@name = 'keywords']">
  <keywords>
    <xsl:for-each select="saxon:tokenize(@content, ',')">
      <xsl:variable name="keyword" select="normalize-space(.)"/>
      <xsl:if test="$keyword != ''">
        <keyword>
          <xsl:value-of select="$keyword"/>
        </keyword>
      </xsl:if>
    </xsl:for-each>
  </keywords>
</xsl:template>

<xsl:template match="h:meta">
  <xsl:if test="@name != ''">
    <data name="{@name}" value="{@content}"/>
  </xsl:if>
</xsl:template>

<!-- body ================================================================ -->

<xsl:template match="h:body">
  <xsl:call-template name="topicBody">
    <xsl:with-param name="mainTopic" select="true()"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="topicBody">
  <xsl:param name="from" select="1"/>
  <xsl:param name="mainTopic" select="false()"/>

  <xsl:variable name="topicBodyName">
    <xsl:choose>
      <xsl:when test="$topic-type = 'concept'">conbody</xsl:when>
      <xsl:when test="$topic-type = 'task' or
                      $topic-type = 'strictTask' or
                      $topic-type = 'generalTask'">taskbody</xsl:when>
      <xsl:when test="$topic-type = 'reference'">refbody</xsl:when>
      <xsl:otherwise>body</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="topicBodyContentName">
    <xsl:choose>
      <xsl:when test="$topic-type = 'task' or
                      $topic-type = 'strictTask'">prereq</xsl:when>
      <xsl:when test="$topic-type = 'generalTask'">steps-informal</xsl:when>
      <xsl:when test="$topic-type = 'reference' or
                      $topic-type = 'concept'">section</xsl:when>
    </xsl:choose>
  </xsl:variable>

  <xsl:element name="{$topicBodyName}">
    <xsl:if test="$mainTopic">
      <xsl:call-template name="processCommonAttributes"/>
    </xsl:if>

    <xsl:choose>
      <xsl:when test="$topicBodyContentName != ''">
        <xsl:element name="{$topicBodyContentName}">
          <xsl:call-template name="topicBodyContent">
            <xsl:with-param name="from" select="$from"/>
          </xsl:call-template>
        </xsl:element>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="topicBodyContent">
          <xsl:with-param name="from" select="$from"/>
        </xsl:call-template>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:element>

  <xsl:if test="$single-topic = 'no'">
    <xsl:apply-templates
        select="./*[self::h:div and starts-with(@class, 'role-section')]"/>
  </xsl:if>
</xsl:template>

<xsl:template name="topicBodyContent">
  <xsl:param name="from" select="1"/>

  <xsl:if test="$pasteFromWordOutput = 'yes'">
    <xsl:processing-instruction name="pfw-begin-body"/>
  </xsl:if>

  <xsl:choose>
    <xsl:when test="$single-topic = 'no'">
      <xsl:variable name="children1"
                    select="./*[position() >= $from]
                               [not(self::h:div and 
                                    (starts-with(@class, 'role-section') or
                                     @class = 'role-footnotes' or
                                     @class = 'role-footnote' or
                                     @class = 'role-endnotes' or
                                     @class = 'role-endnote'))]"/>

      <xsl:call-template name="topicBodyContentNoShortdesc">
        <xsl:with-param name="children" select="$children1"/>
      </xsl:call-template>
    </xsl:when>

    <xsl:otherwise>
      <xsl:variable name="children2"
                    select="./*[not(self::h:div and 
                                    (@class = 'role-footnotes' or
                                     @class = 'role-footnote' or
                                     @class = 'role-endnotes' or
                                     @class = 'role-endnote'))]"/>

      <xsl:call-template name="topicBodyContentNoShortdesc">
        <xsl:with-param name="children" select="$children2"/>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:if test="$pasteFromWordOutput = 'yes'">
    <xsl:processing-instruction name="pfw-end-body"/>
  </xsl:if>
</xsl:template>

<xsl:template name="topicBodyContentNoShortdesc">
  <xsl:param name="children" select="''"/>

  <xsl:choose>
    <xsl:when test="$shortdescClassName != '' and 
                    $children[self::h:p and @class=$shortdescClassName]">
      <xsl:variable name="children2"
        select="sets:difference($children,
                         $children[self::h:p and @class=$shortdescClassName])"/>

      <xsl:choose>
        <xsl:when test="$children2">
          <xsl:apply-templates select="$children2"/>
        </xsl:when>
        <xsl:otherwise><p/></xsl:otherwise>
      </xsl:choose>
    </xsl:when>

    <xsl:otherwise>
      <xsl:choose>
        <xsl:when test="$children">
          <xsl:apply-templates select="$children"/>
        </xsl:when>
        <xsl:otherwise><p/></xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="h:div[starts-with(@class, 'role-section')]">
  <xsl:choose>
    <xsl:when test="$single-topic = 'no'">
      <xsl:variable name="topicName">
        <xsl:call-template name="topicName"/>
      </xsl:variable>

      <xsl:element name="{$topicName}">
        <xsl:call-template name="processCommonAttributes"/>

        <xsl:if test="not(@id)">
          <xsl:attribute name="id">
            <xsl:call-template name="generateTopicId"/>
          </xsl:attribute>
        </xsl:if>

        <xsl:apply-templates select="./*[1]"/> <!-- h1, h2, ... or h6 -->

        <xsl:call-template name="topicShortdesc"/>

        <xsl:call-template name="topicBody">
          <xsl:with-param name="from" select="2"/>
        </xsl:call-template>
      </xsl:element>
    </xsl:when>

    <xsl:otherwise>
      <section>
        <xsl:call-template name="processCommonAttributes"/>

        <xsl:apply-templates select="./*[1]"/> <!-- h1, h2, ... or h6 -->

        <!-- Shortdesc not possible here. -->

        <xsl:variable name="children" select="./*[position() >= 2]"/>
        <xsl:call-template name="topicBodyContentNoShortdesc">
          <xsl:with-param name="children" select="$children"/>
        </xsl:call-template>
      </section>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="topicShortdesc">
  <xsl:if test="$shortdescClassName != ''">
    <xsl:apply-templates select="(./h:p[@class=$shortdescClassName])[1]"/>
  </xsl:if>
</xsl:template>

<!-- body elements ======================================================= -->

<!-- Blocks ========================= -->

<xsl:template match="h:center |
                     h:div[@class != 'role-figure' and
                           @class != 'role-equation' and
                           @class != 'role-bibliography' and
                           not(starts-with(@class, 'role-section')) and
                           @class != 'role-footnotes' and 
                           @class != 'role-footnote' and 
                           @class != 'role-endnotes' and 
                           @class != 'role-endnote']">
  <xsl:call-template name="processDiv"/>
</xsl:template>

<xsl:template name="processDiv">
  <xsl:variable name="divName">
    <xsl:choose>
      <xsl:when test="$single-topic = 'no'">
        <xsl:choose>
          <xsl:when test="$topic-type != 'topic'">sectiondiv</xsl:when>
          <xsl:otherwise>bodydiv</xsl:otherwise>
        </xsl:choose>
      </xsl:when>

      <xsl:otherwise>
        <xsl:if test="$topic-type != 'topic'">
          <xsl:message>Warning: cannot translate XHTML element '<xsl:value-of select="local-name()"/>' to valid DITA when single-topic=yes and topic-type=<xsl:value-of select="$topic-type"/>.</xsl:message>
        </xsl:if>

        <xsl:choose>
          <xsl:when test="$topic-type != 'topic'">sectiondiv</xsl:when>
          <xsl:otherwise>
            <xsl:variable name="section" 
              select="ancestor::h:div[starts-with(@class, 'role-section')]"/>
            <xsl:choose>
              <xsl:when test="$section">sectiondiv</xsl:when>
              <xsl:otherwise>bodydiv</xsl:otherwise>
            </xsl:choose>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:element name="{$divName}">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:div[@class = 'role-figure']">
  <fig>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </fig>
</xsl:template>

<xsl:template match="h:div[@class = 'role-equation']">
  <equation-figure>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </equation-figure>
</xsl:template>

<xsl:template match="h:div[@class = 'role-bibliography']">
  <xsl:choose>
    <xsl:when test="@id != '' or @lang != ''">
      <xsl:call-template name="processDiv"/>
    </xsl:when>

    <xsl:otherwise>
      <!-- Unwrap. -->
      <xsl:apply-templates/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="h:p">
  <xsl:choose>
    <xsl:when test="@class = 'role-inline-wrapper'">
      <xsl:choose>
        <xsl:when test="(parent::h:li or parent::h:dd) and
                        not(preceding-sibling::node())">
          <!-- Unwrap inline wrapper. -->
          <xsl:apply-templates/>
        </xsl:when>

        <xsl:otherwise>
          <p>
            <!-- Inline wrapper: no attributes. -->

            <xsl:if test="parent::h:body and count(../*) = 1">
              <xsl:if test="$pasteFromWordOutput = 'yes'">
                <xsl:processing-instruction name="pfw-inline-wrapper"/>
              </xsl:if>
            </xsl:if>

            <xsl:apply-templates/>
          </p>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>

    <xsl:otherwise>
      <xsl:choose>
        <xsl:when test="parent::h:div[@class = 'role-figure'] and 
                        ./h:img and
                        count(./node()) = 1 and not(@id)">
          <!-- Unwrap. Just keep the image. -->
          <xsl:apply-templates/>
        </xsl:when>

        <xsl:otherwise>
          <p>
            <xsl:call-template name="processCommonAttributes"/>
            <xsl:apply-templates/>
          </p>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="h:p[./mml:math[@display='block']]">
  <!-- We assume here that the p only contains a math block. -->
  <equation-block>
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:for-each select="./mml:math[@display='block']">
      <xsl:call-template name="processMathML"/>
    </xsl:for-each>
  </equation-block>
</xsl:template>

<xsl:template match="h:p[./h:object[@type='application/mathml+xml']]">
  <xsl:choose>
    <xsl:when test="count(./node()) = 1">
      <equation-block>
        <xsl:call-template name="processCommonAttributes"/>

        <xsl:for-each select="./h:object[@type='application/mathml+xml']">
          <xsl:call-template name="processMathMLObject"/>
        </xsl:for-each>
      </equation-block>
    </xsl:when>

    <xsl:otherwise>
      <p>
        <xsl:call-template name="processCommonAttributes"/>
        <xsl:apply-templates/>
      </p>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="h:address |
                     h:h1 | h:h2 | h:h3 | h:h4 | h:h5 | h:h6">
  <p>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </p>
</xsl:template>

<xsl:template match="h:blockquote">
  <lq>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </lq>
</xsl:template>

<xsl:template match="h:pre">
  <xsl:element name="{$pre-element-name}">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<!-- Lists ========================= -->

<xsl:template match="h:ul | h:menu | h:dir">
  <ul>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="processCompactAttribute"/>
    <xsl:apply-templates/>
  </ul>
</xsl:template>

<xsl:template name="processCompactAttribute">
  <xsl:if test="@compact">
    <xsl:attribute name="compact">yes</xsl:attribute>
  </xsl:if>
</xsl:template>

<xsl:template match="h:ol">
  <ol>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="processCompactAttribute"/>
    
    <xsl:if test="@type or @start">
      <xsl:variable name="outputclass">
        <xsl:if test="@type">
          <xsl:choose>
            <xsl:when test="@type = 'a'">lower-alpha</xsl:when>
            <xsl:when test="@type = 'A'">upper-alpha</xsl:when>
            <xsl:when test="@type = 'i'">lower-roman</xsl:when>
            <xsl:when test="@type = 'I'">upper-roman</xsl:when>
            <xsl:otherwise>decimal</xsl:otherwise>
          </xsl:choose>
        </xsl:if>

        <xsl:text> </xsl:text>

        <xsl:if test="@start">
          <xsl:variable name="start">
            <xsl:call-template name="olStart"/>
          </xsl:variable>

          <xsl:choose>
            <xsl:when test="preceding-sibling::h:ol and 
                            @start = $start">continue</xsl:when>
            <xsl:when test="@start != 1">
              <xsl:value-of select="concat('start(', @start, ')')"/>
            </xsl:when>
          </xsl:choose>
        </xsl:if>
      </xsl:variable>

      <xsl:attribute name="outputclass">
        <xsl:value-of select="normalize-space($outputclass)"/>
      </xsl:attribute>
    </xsl:if>

    <xsl:apply-templates/>
  </ol>
</xsl:template>

<xsl:template match="h:dl">
  <!-- Works only if the dl contains sequences comprising
       one or more dt followed by one or more dd. That is, a real dl. -->

  <dl>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="processCompactAttribute"/>

    <xsl:for-each select="./h:dt">
      <xsl:variable name="previous" select="preceding-sibling::*[1]"/>
      <xsl:if test="not($previous) or $previous/self::h:dd">
        <dlentry>
          <xsl:call-template name="dtGroup">
            <xsl:with-param name="head" select="."/>
          </xsl:call-template>

          <xsl:call-template name="ddGroup">
            <xsl:with-param name="head"
                            select="./following-sibling::h:dd[1]"/>
          </xsl:call-template>
        </dlentry>
      </xsl:if>
    </xsl:for-each>
  </dl>
</xsl:template>

<xsl:template name="dtGroup">
  <xsl:param name="head" select="''"/>

  <xsl:for-each select="$head">
    <dt>
      <xsl:call-template name="processCommonAttributes"/>
      <xsl:apply-templates/>
    </dt>
  </xsl:for-each>

  <xsl:variable name="next" select="$head/following-sibling::*[1]"/>
  <xsl:if test="$next/self::h:dt">
    <xsl:call-template name="dtGroup">
      <xsl:with-param name="head" select="$next"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<xsl:template name="ddGroup">
  <xsl:param name="head" select="''"/>

  <xsl:for-each select="$head">
    <dd>
      <xsl:call-template name="processCommonAttributes"/>
      <xsl:apply-templates/>
    </dd>
  </xsl:for-each>

  <xsl:variable name="next" select="$head/following-sibling::*[1]"/>
  <xsl:if test="$next/self::h:dd">
    <xsl:call-template name="ddGroup">
      <xsl:with-param name="head" select="$next"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<!-- Table ========================= -->

<xsl:template match="h:table">
  <xsl:choose>
    <xsl:when test="parent::h:td or parent::h:th or
                    parent::h:div[@class = 'role-footnote' or 
                                  @class = 'role-endnote']">
      <!-- Keep DITA validator quiet. -->
      <p>
        <xsl:call-template name="processTable"/>
      </p>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="processTable"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processTable">
  <table>
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:call-template name="processTableBorderAttribute"/>

    <xsl:if test="./h:caption">
       <xsl:apply-templates select="./h:caption"/>
    </xsl:if>

    <xsl:call-template name="processTgroup"/>
  </table>
</xsl:template>

<!-- Links ========================= -->

<xsl:key name="footnotes"
         match="h:div[@class = 'role-footnote' or @class = 'role-endnote']"
         use="@id"/>

<xsl:template match="h:a">
  <xsl:choose>
    <xsl:when test="@href != ''">
      <xsl:choose>
        <xsl:when test="starts-with(@href, '#') and
                        (@class = 'role-footnote-ref' or 
                         @class = 'role-endnote-ref')">
          <xsl:choose>
            <xsl:when
             test="parent::h:h1[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h2[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h3[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h4[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h5[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h6[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:p[@class = 'role-figcaption'] or
                   parent::h:p[@class = 'role-equation-caption'] or
                   parent::h:caption">
              <!-- Keep DITA validator quiet. -->
              <ph>
                <xsl:call-template name="processFn"/>
              </ph>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="processFn"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <xsl:choose>
            <xsl:when
             test="parent::h:h1[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h2[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h3[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h4[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h5[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:h6[
                     parent::h:div[starts-with(@class, 'role-section')]] or
                   parent::h:p[@class = 'role-figcaption'] or
                   parent::h:p[@class = 'role-equation-caption'] or
                   parent::h:caption">
              <!-- Keep DITA validator quiet. -->
              <ph>
                <xsl:call-template name="processXref"/>
              </ph>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="processXref"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>

    <xsl:otherwise>
      <!-- Not a link nor a footnote. -->
      <xsl:choose>
        <xsl:when test="@name != '' or @id != '' or @lang != ''">
          <ph>
            <xsl:call-template name="processCommonAttributes"/>

            <xsl:if test="@name != '' and not(@id)">
              <xsl:attribute name="id">
                <xsl:value-of select="@name"/>
              </xsl:attribute>
            </xsl:if>

            <xsl:apply-templates/>
          </ph>
        </xsl:when>

        <xsl:otherwise>
          <!-- Unwrap. -->
          <xsl:apply-templates/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processFn">
  <xsl:variable name="footnote"
                select="key('footnotes', substring-after(@href, '#'))"/>
  <xsl:if test="$footnote">
    <xsl:choose>
      <xsl:when
       test="parent::h:em or
             parent::h:strong or
             parent::h:dfn or
             parent::h:abbr or
             parent::h:acronym or
             parent::h:code or
             parent::h:var or
             parent::h:kdb or
             parent::h:samp or
             parent::h:cite or
             parent::h:b or
             parent::h:i or
             parent::h:q or
             parent::h:sub or
             parent::h:sup or
             parent::h:tt or
             parent::h:u or
             parent::h:s or
             parent::h:strike">
        <!-- Keep DITA validator quiet. -->
        <ph>
          <xsl:call-template name="processFootnote">
            <xsl:with-param name="footnote" select="$footnote"/>
          </xsl:call-template>
        </ph>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="processFootnote">
          <xsl:with-param name="footnote" select="$footnote"/>
        </xsl:call-template>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>
</xsl:template>

<xsl:template name="processFootnote">
  <xsl:param name="footnote" select="''"/>

  <fn>
    <xsl:for-each select="$footnote">
      <!-- Do not copy id attribute!
           A footnote that specifies an @id attribute is a use-by-reference
           footnote. Upon output, it does not appear anywhere unless it has
           been referenced using an <xref> with the @type attribute set to
           "fn". -->
      <xsl:call-template name="processLangAttribute"/>
    </xsl:for-each>

    <xsl:apply-templates select="$footnote/node()"/>
  </fn>
</xsl:template>

<xsl:template name="processXref">
  <xref>
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:choose>
      <xsl:when test="starts-with(@href, '#')">
        <xsl:attribute name="href">
          <xsl:call-template name="linkTarget">
            <xsl:with-param name="id"
                            select="substring-after(@href, '#')"/>
          </xsl:call-template>
        </xsl:attribute>
      </xsl:when>
      <xsl:otherwise>
        <xsl:attribute name="href">
          <xsl:value-of select="@href"/>
        </xsl:attribute>
        <xsl:attribute name="scope">external</xsl:attribute>
      </xsl:otherwise>
    </xsl:choose>

    <xsl:apply-templates/>
  </xref>
</xsl:template>

<xsl:key name="linkTargets"
         match="*/@id | h:a/@name"
         use="."/>

<xsl:key name="footnoteRefs"
         match="h:a[starts-with(@href, '#') and
                    (@class = 'role-footnote-ref' or 
                     @class = 'role-endnote-ref')]"
         use="substring-after(@href, '#')"/>

<xsl:template name="linkTarget">
  <xsl:param name="id" select="''"/>

  <!-- linkTarget is an attribute -->
  <xsl:variable name="linkTarget" select="key('linkTargets', $id)[1]"/>

  <xsl:variable name="href">
    <xsl:if test="$single-topic = 'no'">
      <xsl:if test="$linkTarget">
        <xsl:variable name="topicAncestor"
          select="$linkTarget/ancestor::h:div[starts-with(@class, 'role-section')][1]" />

        <xsl:variable name="topicId">
          <xsl:choose>
            <xsl:when test="$topicAncestor">
              <xsl:call-template name="elementId">
                <xsl:with-param name="element" select="$topicAncestor"/>
              </xsl:call-template>
            </xsl:when>

            <xsl:otherwise>
              <!-- The link target is contained in a footnote or endnote. 
                   Use the topic which contains a reference of this 
                   footnote or endnote. -->

              <xsl:variable name="footnote"
                select="$linkTarget/ancestor::h:div[@class = 'role-footnote' or @class = 'role-endnote'][1]"/>
              <xsl:if test="$footnote and $footnote/@id">
                <xsl:variable name="footnoteRef"
                  select="key('footnoteRefs', $footnote/@id)[1]" />
                <xsl:if test="$footnoteRef">
                  <xsl:variable name="footnoteRefAncestor"
                    select="$footnoteRef/ancestor::h:div[starts-with(@class, 'role-section')][1]" />
                  <xsl:if test="$footnoteRefAncestor">
                    <xsl:call-template name="elementId">
                      <xsl:with-param name="element"
                                      select="$footnoteRefAncestor"/>
                    </xsl:call-template>
                  </xsl:if>
                </xsl:if>
              </xsl:if>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>

        <xsl:if test="$topicId != ''">
          <xsl:choose>
            <!-- linkTarget is an attribute, hence parent::h:div. -->
            <xsl:when test="$linkTarget/parent::h:div[starts-with(@class, 'role-section')]">
              <xsl:value-of select="concat('#', $topicId)"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="concat('#', $topicId, '/', $id)"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:if>

      </xsl:if>
    </xsl:if>
  </xsl:variable>

  <xsl:choose>
    <xsl:when test="$href != ''">
      <xsl:value-of select="$href"/>
    </xsl:when>

    <xsl:otherwise>
      <xsl:variable name="rootId">
        <xsl:call-template name="elementId">
          <xsl:with-param name="element" select="/*"/>
        </xsl:call-template>
      </xsl:variable>

      <xsl:choose>
        <xsl:when test="$linkTarget/parent::h:html">
          <xsl:value-of select="concat('#', $rootId)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="concat('#', $rootId, '/', $id)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="elementId">
  <xsl:param name="element" select="''"/>

  <xsl:choose>
    <xsl:when test="not($element/parent::*) and 
                    $root-topic-id != '' and $root-topic-id != '???'">
      <xsl:value-of select="$root-topic-id"/>
    </xsl:when>
    <xsl:when test="$element/@id">
      <xsl:value-of select="$element/@id"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="generate-id($element)"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Image ========================= -->

<xsl:template match="h:img">
  <xsl:choose>
    <xsl:when
     test="parent::h:em or
           parent::h:strong or
           parent::h:dfn or
           parent::h:abbr or
           parent::h:acronym or
           parent::h:code or
           parent::h:var or
           parent::h:kdb or
           parent::h:samp or
           parent::h:cite or
           parent::h:b or
           parent::h:i or
           parent::h:q or
           parent::h:sub or
           parent::h:sup or
           parent::h:tt or
           parent::h:u or
           parent::h:s or
           parent::h:strike">
      <!-- Keep DITA validator quiet. -->
      <ph>
        <xsl:call-template name="processImage"/>
      </ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="processImage"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processImage">
  <image>
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:attribute name="href">
      <xsl:value-of select="@src"/>
    </xsl:attribute>

    <xsl:for-each select="@width | @height">
      <!-- Percentage not supported. -->
      <xsl:if test="number(.) > 0">
        <xsl:copy/>
      </xsl:if>
    </xsl:for-each>

    <xsl:for-each select="@alt">
      <xsl:copy/>
    </xsl:for-each>
  </image>
</xsl:template>

<!-- MathML ========================= -->

<xsl:template match="h:object[@type='application/mathml+xml']">
  <equation-inline>
    <xsl:call-template name="processMathMLObject"/>
  </equation-inline>
</xsl:template>

<xsl:template name="processMathMLObject">
  <mathml>
    <mathmlref href="{@data}"/>
  </mathml>
</xsl:template>

<xsl:template match="mml:math">
  <equation-inline>
    <xsl:call-template name="processMathML"/>
  </equation-inline>
</xsl:template>

<xsl:template name="processMathML">
  <mathml>
    <xsl:apply-templates select="." mode="math"/>
  </mathml>
</xsl:template>

<!-- The DITA 1.3 *DTD* only supports the 'm:' prefix. -->

<xsl:template match="mml:*" mode="math">
  <xsl:element name="{concat('m:', local-name())}"
               xmlns:m="http://www.w3.org/1998/Math/MathML">
    <xsl:apply-templates select="@*|node()" mode="math"/>
  </xsl:element>
</xsl:template>

<xsl:template match="@*|node()" mode="math">
  <xsl:copy><xsl:apply-templates select="@*|node()" mode="math"/></xsl:copy>
</xsl:template>

<!-- Inlines ========================= -->

<xsl:template match="h:em">
  <i>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </i>
</xsl:template>

<xsl:template match="h:strong">
  <b>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </b>
</xsl:template>

<xsl:template match="h:dfn">
  <term>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </term>
</xsl:template>

<xsl:template match="h:abbr | h:acronym">
  <keyword>
    <xsl:call-template name="processCommonAttributes"/>
    <!-- A keyword can basically only contain text. -->
    <xsl:value-of select="."/>
  </keyword>
</xsl:template>

<xsl:template match="h:code">
  <codeph>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </codeph>
</xsl:template>

<xsl:template match="h:var">
  <varname>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </varname>
</xsl:template>

<xsl:template match="h:kbd">
  <userinput>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </userinput>
</xsl:template>

<xsl:template match="h:samp">
  <systemoutput>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </systemoutput>
</xsl:template>

<xsl:template match="h:cite |
                     h:b |
                     h:i |
                     h:li |
                     h:q |
                     h:sub |
                     h:sup |
                     h:tt |
                     h:u">
  <xsl:element name="{local-name()}">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:ruby">
  <ph>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:value-of select="."/>
  </ph>
</xsl:template>

<xsl:template match="h:hr |
                     h:br">
  <xsl:text> </xsl:text>
</xsl:template>

<xsl:template match="h:span[@class = 'role-index-term' or 
                            @class = 'role-index-term-start' or 
                            @class = 'role-index-term-end']">
  <xsl:choose>
    <xsl:when test="parent::h:h1 |
                    parent::h:h2 |
                    parent::h:h3 |
                    parent::h:h4 |
                    parent::h:h5 |
                    parent::h:h6 |
                    parent::h:dt |
                    parent::h:caption |
                    parent::h:p[@class = 'role-figcaption']">
      <!-- Keep DITA validator quiet. -->
      <ph>
        <xsl:call-template name="processIndexterm"/>
      </ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="processIndexterm"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processIndexterm">
    <xsl:choose>
      <xsl:when test="@class = 'role-index-term-end'">
        <indexterm end="{concat('__RANGE', substring-after(@id, '__IDX_E_'))}"/>
      </xsl:when>

      <xsl:when test="@class = 'role-index-term-start'">
        <xsl:variable name="start" 
          select="concat('__RANGE', substring-after(@id, '__IDX_S_'))"/>
        <xsl:call-template name="doProcessIndexterm">
          <xsl:with-param name="start" select="$start"/>
        </xsl:call-template>
      </xsl:when>

      <xsl:otherwise>
        <xsl:call-template name="doProcessIndexterm"/>
      </xsl:otherwise>
    </xsl:choose>
</xsl:template>

<xsl:template name="doProcessIndexterm">
  <xsl:param name="start" select="''"/>

  <!-- First term ==================== -->
  <indexterm>
    <xsl:if test="$start != '' and 
                  not(./h:span[@class = 'role-index-term-2'])">
      <xsl:attribute name="start">
        <xsl:value-of select="$start"/>
      </xsl:attribute>
    </xsl:if>

    <xsl:value-of select="./h:span[@class = 'role-index-term-1']"/>

    <xsl:call-template name="indexTermSortAs">
      <xsl:with-param name="sortAs" 
        select="./h:span[@class = 'role-index-term-1']/@title"/>
    </xsl:call-template>

    <xsl:choose>
      <xsl:when test="./h:span[@class = 'role-index-term-2']">
        <!-- Second term ==================== -->
        <indexterm>
          <xsl:if test="$start != '' and 
                        not(./h:span[@class = 'role-index-term-3'])">
            <xsl:attribute name="start">
              <xsl:value-of select="$start"/>
            </xsl:attribute>
          </xsl:if>

          <xsl:value-of select="./h:span[@class = 'role-index-term-2']"/>

          <xsl:call-template name="indexTermSortAs">
            <xsl:with-param name="sortAs" 
              select="./h:span[@class = 'role-index-term-2']/@title"/>
          </xsl:call-template>

          <xsl:choose>
            <xsl:when test="./h:span[@class = 'role-index-term-3']">
              <!-- Third term ==================== -->
              <indexterm>
                <xsl:if test="$start != ''">
                  <xsl:attribute name="start">
                    <xsl:value-of select="$start"/>
                  </xsl:attribute>
                </xsl:if>

                <xsl:value-of select="./h:span[@class = 'role-index-term-3']"/>

                <xsl:call-template name="indexTermSortAs">
                  <xsl:with-param name="sortAs" 
                    select="./h:span[@class = 'role-index-term-3']/@title"/>
                </xsl:call-template>

                <xsl:call-template name="indexTermSee"/>
              </indexterm>
            </xsl:when>

            <xsl:otherwise>
              <xsl:call-template name="indexTermSee"/>
            </xsl:otherwise>
          </xsl:choose>
        </indexterm>
      </xsl:when>

      <xsl:otherwise>
        <xsl:call-template name="indexTermSee"/>
      </xsl:otherwise>
    </xsl:choose>
  </indexterm>
</xsl:template>

<xsl:template name="indexTermSortAs">
  <xsl:param name="sortAs" select="''"/>

  <xsl:if test="$sortAs">
    <index-sort-as><xsl:value-of select="$sortAs"/></index-sort-as>
  </xsl:if>
</xsl:template>

<xsl:template name="indexTermSee">
  <xsl:if test="./h:span[@class = 'role-index-term-see']">
    <index-see>
      <xsl:value-of select="./h:span[@class = 'role-index-term-see']"/>
    </index-see>
  </xsl:if>
</xsl:template>

<xsl:template match="h:s | h:strike">
  <line-through>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </line-through>
</xsl:template>

<xsl:template match="h:big |
                     h:font |
                     h:span |
                     h:small">
  <xsl:choose>
    <xsl:when test="@id != '' or @lang != ''">
      <ph>
        <xsl:call-template name="processCommonAttributes"/>
        <xsl:apply-templates/>
      </ph>
    </xsl:when>

    <xsl:otherwise>
      <!-- Unwrap. -->
      <xsl:apply-templates/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="h:ins">
  <!-- Unwrap. -->
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="h:del"/>

<!-- Attributes ========================================================== -->

<xsl:template name="processCommonAttributes">
  <xsl:call-template name="processIdAttribute"/>
  <xsl:call-template name="processLangAttribute"/>
</xsl:template>

<xsl:template name="processIdAttribute">
  <xsl:for-each select="@id">
    <xsl:copy/>
  </xsl:for-each>
</xsl:template>

<xsl:template name="processLangAttribute">
  <xsl:if test="@lang != ''">
    <xsl:attribute name="xml:lang">
      <xsl:value-of select="@lang"/>
    </xsl:attribute>
  </xsl:if>
</xsl:template>

<!-- Not translated ==========================================================
applet
area
basefont
base
bdo
button
fieldset
form
iframe
input
isindex
label
legend
link
map
noframes
noscript
object
optgroup
option
param
script
select
style
textarea
========================================================================== -->

<xsl:template match="*">
  <xsl:message>Internal error: XHTML element '<xsl:value-of select="local-name(.)"/>' has not been translated to DITA. Please report a bug to xmleditor-support@xmlmind.com.</xsl:message>

  <foreign>
    <xsl:element name="{local-name()}" namespace="http://www.w3.org/1999/xhtml">
      <xsl:apply-templates select="@*|node()" mode="asis"/>
    </xsl:element>
  </foreign>
</xsl:template>

<xsl:template match="@*|node()" mode="asis">
  <xsl:copy><xsl:apply-templates select="@*|node()" mode="asis"/></xsl:copy>
</xsl:template>

</xsl:stylesheet>
